﻿#include "precompiled.h"
#include "ResourceManager.h"
#include "common.h"

#include "Mesh.h"
#include "Material.h"
#include "Texture.h"
#include "Texture1D.h"

using namespace DirectX;

namespace RTCam {

ResourceManager::ResourceManager(void) :
	m_meshDictionary(),
	m_materialDictionary()
{
}


ResourceManager::~ResourceManager(void)
{
}

void ResourceManager::Initialize(_In_ ID3D11Device1* device)
{
	CreateDefaultMeshes(device);
	CreateDefaultMaterials(device);
	CreateDefaultTextures(device);
}

void ResourceManager::CreateDefaultMeshes(_In_ ID3D11Device1* device)
{
	XMFLOAT3 XPos(1, 0, 0);
	XMFLOAT3 XNeg(-1, 0, 0);
	XMFLOAT3 YPos(0, 1, 0);
	XMFLOAT3 YNeg(0, -1, 0);
	XMFLOAT3 ZPos(0, 0, 1);
	XMFLOAT3 ZNeg(0, 0, -1);

	XMFLOAT3 V0Pos(-0.5f, -0.5f, -0.5f);
	XMFLOAT3 V1Pos(-0.5f, -0.5f,  0.5f);
	XMFLOAT3 V2Pos(-0.5f,  0.5f, -0.5f);
	XMFLOAT3 V3Pos(-0.5f,  0.5f,  0.5f);
	XMFLOAT3 V4Pos( 0.5f, -0.5f, -0.5f);
	XMFLOAT3 V5Pos( 0.5f, -0.5f,  0.5f);
	XMFLOAT3 V6Pos( 0.5f,  0.5f, -0.5f);
	XMFLOAT3 V7Pos( 0.5f,  0.5f,  0.5f);

	XMFLOAT4 V0Col(0.0f, 0.0f, 0.0f, 1.0f);
	XMFLOAT4 V1Col(0.0f, 0.0f, 1.0f, 1.0f);
	XMFLOAT4 V2Col(0.0f, 1.0f, 0.0f, 1.0f);
	XMFLOAT4 V3Col(0.0f, 1.0f, 1.0f, 1.0f);
	XMFLOAT4 V4Col(1.0f, 0.0f, 0.0f, 1.0f);
	XMFLOAT4 V5Col(1.0f, 0.0f, 1.0f, 1.0f);
	XMFLOAT4 V6Col(1.0f, 1.0f, 0.0f, 1.0f);
	XMFLOAT4 V7Col(0.5f, 0.5f, 0.5f, 1.0f);

	// The cube
	VertexPositionNormalColor cubeVertices[] =
	{
		// -x
		VertexPositionNormalColor(V0Pos, XNeg, V0Col), // 0
		VertexPositionNormalColor(V1Pos, XNeg, V1Col), // 1
		VertexPositionNormalColor(V2Pos, XNeg, V2Col), // 2
		VertexPositionNormalColor(V3Pos, XNeg, V3Col), // 3
		// +x
		VertexPositionNormalColor(V4Pos, XPos, V4Col), // 4
		VertexPositionNormalColor(V5Pos, XPos, V5Col), // 5
		VertexPositionNormalColor(V6Pos, XPos, V6Col), // 6
		VertexPositionNormalColor(V7Pos, XPos, V7Col), // 7
		// -y
		VertexPositionNormalColor(V0Pos, YNeg, V0Col), // 0 8
		VertexPositionNormalColor(V1Pos, YNeg, V1Col), // 1 9
		VertexPositionNormalColor(V4Pos, YNeg, V4Col), // 4 10
		VertexPositionNormalColor(V5Pos, YNeg, V5Col), // 5 11
		// +y
		VertexPositionNormalColor(V2Pos, YPos, V2Col), // 2 12
		VertexPositionNormalColor(V3Pos, YPos, V3Col), // 3 13
		VertexPositionNormalColor(V6Pos, YPos, V6Col), // 6 14
		VertexPositionNormalColor(V7Pos, YPos, V7Col), // 7 15
		// -z
		VertexPositionNormalColor(V0Pos, ZNeg, V0Col), // 0 16
		VertexPositionNormalColor(V2Pos, ZNeg, V2Col), // 2 17
		VertexPositionNormalColor(V4Pos, ZNeg, V4Col), // 4 18
		VertexPositionNormalColor(V6Pos, ZNeg, V6Col), // 6 19
		// +z
		VertexPositionNormalColor(V1Pos, ZPos, V1Col), // 1 20
		VertexPositionNormalColor(V3Pos, ZPos, V3Col), // 3 21
		VertexPositionNormalColor(V5Pos, ZPos, V5Col), // 5 22
		VertexPositionNormalColor(V7Pos, ZPos, V7Col), // 7 23
	};
	uint16_t cubeIndices[] = 
	{
		0,1,2, // -x
		1,3,2,

		4,6,5, // +x
		5,6,7,

		8,11,9, // -y
		8,10,11,

		12,15,14, // +y
		12,13,15,

		16,19,18, // -z
		16,17,19,

		20,23,21, // +z
		20,22,23,
	};

	m_meshColorCube = shared_ptr<Mesh>(new Mesh("ColorCube"));
	m_meshColorCube->Initialize(device, cubeVertices, ARRAYSIZE(cubeVertices), cubeIndices, ARRAYSIZE(cubeIndices));

	// Make standard (gray) cube
	XMFLOAT4 gray(0.5f, 0.5f, 0.5f, 1);

	for(auto& vertex: cubeVertices) {
		vertex.color = gray;
	}
	m_meshCube = shared_ptr<Mesh>(new Mesh("Cube"));
	m_meshCube->Initialize(device, cubeVertices, ARRAYSIZE(cubeVertices), cubeIndices, ARRAYSIZE(cubeIndices));
}

void ResourceManager::CreateDefaultMaterials(_In_ ID3D11Device1* device)
{
	m_matDefault = shared_ptr<Material>(new Material("DefaultMat"));

	m_matEmissive = shared_ptr<Material>(new Material("EmissiveMat"));
	m_matEmissive->m_emission.x = 250;
	m_matEmissive->m_emission.y = 250;
	m_matEmissive->m_emission.z = 250;

	m_matMagenta = shared_ptr<Material>(new Material("Magenta"));
	// TODO: Extend materials when vertex colors are associated with vertex groups instead of individual vertices.
}


void ResourceManager::CreateDefaultTextures( _In_ ID3D11Device1* device )
{
	m_texBokeh = shared_ptr<Texture>(new Texture("BokehTex"));
	m_texBokeh->InitFromDDS(device, "Textures\\Hexagon.dds");

	m_texPositiveSA = shared_ptr<Texture>(new Texture("PositiveSATex"));
	m_texPositiveSA->InitFromDDS(device, "Textures\\PositiveSA.dds");

	m_texZeroSA = shared_ptr<Texture>(new Texture("ZeroSATex"));
	m_texZeroSA->InitFromDDS(device, "Textures\\ZeroSA.dds");

	m_texNegativeSA = shared_ptr<Texture>(new Texture("NegativeSATex"));
	m_texNegativeSA->InitFromDDS(device, "Textures\\NegativeSA.dds");

	// Create the default texture used for lens/film parameters
	const int arraySize = 23; // Film radius ranges from 0mm to 22mm
	float texValues[arraySize];
	ZeroMemory(texValues, sizeof(float) * arraySize);

	m_texDefault1D = shared_ptr<Texture1D>(new Texture1D("Default 1D texture"));
	m_texDefault1D->InitFromChart(device, texValues, arraySize);

	// Vignetting needs its own texture with values set to 100
	for(auto& value : texValues) {
		value = 100;
	}
	m_texDefaultVignetting = shared_ptr<Texture1D>(new Texture1D("Default Vignetting"));
	m_texDefaultVignetting->InitFromChart(device, texValues, arraySize);
}


void ResourceManager::AddMesh(const shared_ptr<Mesh>& mesh)
{
	THROW_UNIMPLEMENTED_EXCEPTION();

	// Make sure a mesh with the given name doesn't already exist
	const string& meshName = mesh->GetName();
	auto nameExists = m_meshDictionary.find(meshName);
	if(nameExists == m_meshDictionary.end()) {
		string errorMsg("A mesh with the name already exists");
		BreakAndThrow(errorMsg.c_str());
	}
	
	std::pair<string, shared_ptr<Mesh>> newPair(mesh->GetName(), mesh);
	m_meshDictionary.insert(newPair);
}

weak_ptr<Mesh> ResourceManager::GetMesh(const string& name)
{
	THROW_UNIMPLEMENTED_EXCEPTION();
	return weak_ptr<Mesh>();
}

} // end namespace